-- Adds an "Open Recipe UI" option when right-clicking a plausible cooking base item
-- (e.g. empty roasting pan, bowl, pot, saucepan).
--
-- Inputs (from event):
--   playerNum: integer            -- player index
--   context:   ISContextMenu      -- the context menu being built
--   items:     table              -- mixed list of InventoryItems / wrapper tables
--
-- This step intentionally uses a simple heuristic to decide what is a "cooking base".
-- We'll refine this later to mirror vanilla evolved recipe logic exactly.

require "ISUI/ISContextMenu"
require "UI/EvolvedRecipeUI"

-------------------------------------------------
-- Helpers
-------------------------------------------------

---Resolve the underlying InventoryItem from the mixed `items` array that
---OnFillInventoryObjectContextMenu passes in.
---@param items table
---@return InventoryItem|nil
local function BC_getPrimaryItemFromContext(items)
    if not items or #items == 0 then
        return nil
    end

    local first = items[1]
    -- In many vanilla handlers, the item is wrapped: { items = { InventoryItem, ... }, ... }
    if not instanceof(first, "InventoryItem") and first.items and #first.items > 0 then
        first = first.items[1]
    end

    if instanceof(first, "InventoryItem") then
        return first
    end

    return nil
end

---Temporary Hack (Build 42.x) — reason: we don't yet mirror vanilla's evolved recipe checks.
---This is a conservative heuristic for "likely cooking base" items.
---
---@param item InventoryItem
---@return boolean
local function BC_isLikelyCookingBase(item)
    if not item then
        return false
    end

    -- Empty roasting pans, bowls, pots, saucepans, etc.
    local itemType = item:getType()
    if itemType == "RoastingPan" or
       itemType == "Pan" or
       itemType == "Pot" or
       itemType == "Saucepan" or
       itemType == "Bowl" then
        return true
    end

    -- Many prepared dishes are actual Food items; allow those too.
    -- (e.g. PanFriedVeg, PanFriedSteak, StewPot, etc.)
    if instanceof(item, "Food") then
        return true
    end

    -- You can expand this list later (e.g., modded containers, kettles, etc.).
    return false
end

-------------------------------------------------
-- Context menu hook
-------------------------------------------------

---OnFillInventoryObjectContextMenu handler for Better Cooking UI.
---
---@param playerNum integer
---@param context ISContextMenu
---@param items table
local function BC_OnFillInventoryObjectContextMenu(playerNum, context, items)
    local playerObj = getSpecificPlayer(playerNum)
    if not playerObj or not context or not items then
        return
    end

    local baseItem = BC_getPrimaryItemFromContext(items)
    if not baseItem then
        return
    end

    -- Placeholder gating: only show on plausible cooking base items for now.
    -- Later we can refine this by calling the same helpers vanilla uses for
    -- "Add Ingredient / Add Spice" evolved recipe checks.
    if not BC_isLikelyCookingBase(baseItem) then
        return
    end

    -- Add our "Open Recipe UI" option.
    local optionText = getTextOrNull("IGUI_BCUi_OpenRecipeUI") or "Open Recipe UI"
    context:addOptionOnTop(
        optionText,
        playerObj,
        function(player, item)
            BC_EvolvedRecipeUI.openFor(player, item)
        end,
        baseItem
    )
end

-- Events.OnFillInventoryObjectContextMenu.Add(BC_OnFillInventoryObjectContextMenu)
